/*
 *
 * Copyright 1998-1999, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Kinis L. Meyer with M. D. McNally 
 *          and Andrew Lumsdaine
 *
 * This file is part of the Notre Dame LAM implementation of MPI.
 *
 * You should have received a copy of the License Agreement for the
 * Notre Dame LAM implementation of MPI along with the software; see
 * the file LICENSE.  If not, contact Office of Research, University
 * of Notre Dame, Notre Dame, IN 46556.
 *
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 *
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.  
 *
 * Additional copyrights may follow.
 *
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	NJN
 *
 *	$Id: shm.sysv.c,v 6.7 1999/07/28 00:32:04 jsquyres Exp $
 *
 *	Function:	- sysv transport specific low-level routines
 */

#include <lam_config.h>

#include <assert.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/uio.h>

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>
#include <rpisys.h>
#include <terror.h>
#include <typical.h>
#include <t_types.h>

/*
 * global functions
 */
int			_shm_serverinit();
int			_shm_clientinit();
int			_shm_cleanup();
int			_shm_global_cleanup();

/*
 * external functions
 */
extern void		lam_register_objects();
extern void		lam_deregister_object();

/*
 * private variables
 *
 * Per connection semaphores 0 and 1 are used for cleanup control.
 * Semaphores 2, 3, 4 and 5 are used for transfer buffer locking as follows.
 *
 * 2 - server write lock, client read unlock
 * 3 - client read lock, server write unlock
 * 4 - client write lock, server read unlock
 * 5 - server read lock, client write unlock
 */
static struct sembuf	op_lock[5] = {
				{ 3, -1, 0 },
				{ 4, -1, 0 },
				{ 5, -1, 0 },
				{ 2, -1, 0 },
				{ 0, -1, 0 }
			};

static struct sembuf	op_trylock[4] = {
				{ 3, -1, IPC_NOWAIT },
				{ 4, -1, IPC_NOWAIT },
				{ 5, -1, IPC_NOWAIT },
				{ 2, -1, IPC_NOWAIT },
			};


static struct sembuf	op_unlock[5] = {
				{ 2, 1, 0 },
				{ 5, 1, 0 },
				{ 4, 1, 0 },
				{ 3, 1, 0 },
				{ 0, 1, 0 },
			};

/*
 * global semaphores
 */
static struct sembuf	exit_lock = { 0, -1, 0 };
static struct sembuf	exit_unlock = { 0, 1, 0 };
static struct sembuf	exit_op = { 1, -1, IPC_NOWAIT };

static struct sembuf	sop;

#if LAM_HAVE_UNION_SEMUN
static union semun	semctl_arg;
#else
static union {
	int		val;
	struct semid_ds	*buf;
	unsigned short	*array;
} semctl_arg;
#endif

/*
 *	_shm_clientinit
 *
 *	Function:	- client side shared memory initialization
 *	Accepts:	- process
 *			- LAM msg containing server info
 *	Returns:	- 0 or LAMERROR
 */
int
_shm_clientinit(ps, msg)

struct c2c_proc		*ps;
struct nmsg		*msg;

{
	ps->cp_ppsem = (int) msg->nh_data[3];
/*
 * Initialize the semaphore operations.
 */
	ps->cp_lop = op_lock;
	ps->cp_top = op_trylock;
	ps->cp_uop = op_unlock;

	return(0);
}

/*
 *	_shm_serverinit
 *
 *	Function:	- server side shared memory initialization
 *	Accepts:	- process
 *			- LAM msg to fill with info for client
 *	Returns:	- 0 or LAMERROR
 */
int
_shm_serverinit(ps, msg)

struct c2c_proc		*ps;
struct nmsg		*msg;

{
	char		objs[2][32];
	int		semid;
/*
 * Initialize the semaphore operations.
 */
	ps->cp_lop = op_lock + 2;
	ps->cp_top = op_trylock + 2;
	ps->cp_uop = op_unlock + 2;
/*
 * Create the semaphores.
 */
	if ((semid = semget(IPC_PRIVATE, 6, 0600 | IPC_CREAT)) < 0) {
		errno = ESEMCREATE;
		return(LAMERROR);
	}
/*
 * Register id's for cleanup.
 */
	sprintf(objs[1], "%d", semid);
	lam_register_objects(1, 's', objs[1]);
/*
 * Initilize semaphores.
 */
	ps->cp_ppsem = semid;

	semctl_arg.val = 1;
	if (semctl(semid, 0, SETVAL, semctl_arg) < 0) return(LAMERROR);
	semctl_arg.val = 2;
	if (semctl(semid, 1, SETVAL, semctl_arg) < 0) return(LAMERROR);

	semctl_arg.val = 1;
	if (semctl(semid, 2, SETVAL, semctl_arg) < 0) return(LAMERROR);
	semctl_arg.val = 0;
	if (semctl(semid, 3, SETVAL, semctl_arg) < 0) return(LAMERROR);

	semctl_arg.val = 1;
	if (semctl(semid, 4, SETVAL, semctl_arg) < 0) return(LAMERROR);
	semctl_arg.val = 0;
	if (semctl(semid, 5, SETVAL, semctl_arg) < 0) return(LAMERROR);
/*
 * Set information to pass to client.
 */
	msg->nh_data[3] = (int4) semid;

	return(0);
}

/*
 *	_shm_cleanup
 *
 *	Function:	- clean up a process's per conection shared memory
 *			  structures
 *	Accepts:	- process
 *	Returns:	- 0 or LAMERROR
 */
int
_shm_cleanup(ps)

struct c2c_proc		*ps;

{
	char		obj[32];
	int		val;			/* cleanup counter */
/*
 * Check if the other side of the connection has cleaned up.
 */
	sop.sem_num = 1;
	sop.sem_op = -1;
	sop.sem_flg = IPC_NOWAIT;
	semctl_arg.val = 0;

	if (semop(ps->cp_ppsem, op_lock + 4, 1) < 0) return(LAMERROR);
	if (semop(ps->cp_ppsem, &sop, 1) < 0) return(LAMERROR);

	val = semctl(ps->cp_ppsem, 1, GETVAL, semctl_arg);
	if (val < 0) return(LAMERROR);

	if (semop(ps->cp_ppsem, op_unlock + 4, 1) < 0) return(LAMERROR);

	if (val == 0) {
/*
 * The other side of the shared area has already cleaned up.
 * Delete the per process pair semaphores and deregister.
 */
		semctl(ps->cp_ppsem, 0, IPC_RMID, semctl_arg);

		sprintf(obj, "%d", ps->cp_ppsem);
		lam_deregister_object('s', obj);
	}

	return(0);
}

/*
 *	_shm_global_cleanup
 *
 *	Function:	- clean up a process's global shared memory structures
 *	Accepts:	- process
 *	Returns:	- 0 or LAMERROR
 */
int
_shm_global_cleanup(ps)

struct c2c_proc		*ps;

{
	char		obj[32];
	int		val;			/* cleanup counter */

	if (semop(ps->cp_sem, &exit_lock, 1) < 0) return(LAMERROR);
	if (semop(ps->cp_sem, &exit_op, 1) < 0) return(LAMERROR);

	semctl_arg.val = 0;
	val = semctl(ps->cp_sem, 1, GETVAL, semctl_arg);
	if (val < 0) return(LAMERROR);

	shmdt(_shm_membase);

	if (semop(ps->cp_sem, &exit_unlock, 1) < 0) return(LAMERROR);

	if (val == 0) {
/*
 * The other side of the shared area has already cleaned up so
 * we can delete the semaphores and deregister the shared structures.
 */
		semctl(ps->cp_sem, 0, IPC_RMID, semctl_arg);
		shmctl(ps->cp_shm, IPC_RMID, (struct shmid_ds *) 0);

		sprintf(obj, "%d", ps->cp_sem);
		lam_deregister_object('s', obj);
		sprintf(obj, "%d", ps->cp_shm);
		lam_deregister_object('m', obj);
	}

	return(0);
}

/*
 *	_shm_readlock
 *
 *	Function:	- obtain a lock for reading from a process
 *	Accepts:	- process to read from
 *	Returns:	- 0 or LAMERROR
 */
int
_shm_readlock(p)

struct c2c_proc		*p;

{
	do {
		if (semop((p)->cp_ppsem, (p)->cp_lop, 1) == 0) {
			return(0);
		} else if (errno != EINTR) {
			return(LAMERROR);
		}
	} while (1);
}

/*
 *	_shm_writelock
 *
 *	Function:	- obtain a lock for writing to a process
 *	Accepts:	- process to write to
 *	Returns:	- 0 or LAMERROR
 */
int
_shm_writelock(p)

struct c2c_proc		*p;

{
	do {
		if (semop((p)->cp_ppsem, (p)->cp_lop + 1, 1) == 0) {
			return(0);
		} else if (errno != EINTR) {
			return(LAMERROR);
		}
	} while (1);
}
