#include "wily.h"
#include "data.h"
#include <errno.h>

Data		*dataroot;

Text*
data_body(Data*d) {
	return d->t;
}

Text*
data_tag(Data*d){
	return d->tag;
}

/* Return the names in directory 'd', or 0 */
char**
data_names(Data*d)
{
	return d?d->names:0;
}

/* Return true if 'd' should have its resize box filled in */
Bool
data_isdirty(Data *d) {
	return d && NEEDSBACKUP(d);
}

/* Copy the path associated with 'd' into 'dest' */
void
data_path(Data*d, char*dest)
{
	strcpy(dest, d ? d->path : wilydir);
}

/* Store the context for 'd' into 's' */
void
data_context(Data*d, char*s) {
	data_path(d,s);
	if( (s=strrchr(s,'/')) )
		s[1]='\0';
}

/* Write all dirty windows.  Return 0 for success. */
int
data_putall(void)
{
	Data	*d;
	int	retval = 0;

	for(d=dataroot; d; d=d->next)
		if(NEEDSBACKUP(d))
			if (data_put(d,0))
				retval = -1;
	return retval;
}

/*
 * Backup all dirty files.  Send messages to stderr, in case wily itself
 * is going down soon.  Return 0 unless we failed to backup something.
 */
int
data_backupall(void)
{
	Data	*d;
	int	retval= 0;

	for(d = dataroot; d; d = d->next) {
		if (data_backup(d)) {
			retval = 1;
			fprintf(stderr,"backup of %s failed\n", d->path);
		}
	}
	return retval;
}

/*
 * Write the contents of 'd' to 'path'.
 * Return 0 if we've successfully written 'd' to its label.
 */
int
data_put(Data *d, char *path)
{
	if(!path)
		path = d->path;

	if(!text_write(d->t, path) && STRSAME(path, d->path)) {
		if(!d->names) {
			tag_rmtool(d->tag, "Put");
			undo_mark(d->t);
		}
		return 0;
	} else {
		return -1; /* write failed, or different path */
	}
}

/* 
 * Return 'true' iff 'big' and 'small' are identical, or
 * if 'big' == 'small' + a trailing slash.  'small' is of
 * length 'len'.
 */
static Bool
equal_modulo_slash(char*big, char*small, int len) {
	return !strncmp(big, small, len) &&
	(big[len]==0 || (big[len]=='/' && big[len+1]==0));
}

/*
 * Return pointer to View with same 'path', or null.
 * 'path' is assumed to be the output of realpath, i.e. a canonical path.
*/
View *
data_find(char*path)
{
	Data	*d;
	int	len;
	
	len = strlen(path);

	for(d=dataroot; d; d=d->next)
		if(equal_modulo_slash(d->path, path, len))
			return text_view(d->t);
	return 0;
}

/* Delete 'd' if we can do so without permanently losing
 * data, and return 0.  If there's a problem, return 1.
 */
int
data_del(Data*d)
{
	Data	**ptr;

	if(data_backup(d))
		return 1;
	
	for(ptr = &dataroot; *ptr != d; ptr = &((*ptr)->next))
		;
	*ptr = d->next;
	dirnames_free(d->names);
	if(d->backupto)
		free(d->backupto);
	free(d);
	return 0;
}

void
data_setlabel(Data*d, char *s) {
	assert(d);

	strcpy(d->label, s);
	pathexpand(s, 0, d->path);
	
	if(isdir(d->path)) {
		add_slash(d->path);
		tag_addtool(d->tag, "Get");
	} else {
		tag_addtool(d->tag, "Put");
	}
}

/* Backup 'd' if necessary.  Return 0 for success. */
int
data_backup(Data *d)
{
	Path	fname;

	if (!NEEDSBACKUP(d))
		return 0;

	if( (backup_name(d->backupto, fname)) )
		return -1;
	if(text_write(d->t, fname))
		return -1;

	errno = 0;
	diag(fname, "backup %s %s", mybasename(fname), d->backupto);
	return 0;
}

/* Record that in emergencies, 'd' should be saved, noting
 * that it is a copy of 'bfile'
 */
void
data_setbackup(Data *d, char*bfile)
{
	if(d->backupto)
		free(d->backupto);
	if(bfile && !strstr(bfile, "+Errors")) {
		d->backupto = strdup(bfile);
		text_setneedsbackup(d->t, true);
	} else {
		d->backupto = 0;
		text_setneedsbackup(d->t, false);
	};
}

Data *
data_findid(Id id)
{
	Data	*d;

	for(d=dataroot; d; d=d->next)
		if(d->id==id)
			break;
	return d;
}

