/*
** Jabby (a C library for Jabber protocol)
** Copyright (c) 2002 Hubert Sokoowski <who_ami@tlen.pl>
**
** This code is free software; you can redistribute it and/or
** modify it under the terms of the GNU General Public License.
**
*/

#include "jabby.h"
#include "events.h"
#include "parser.h"

#include <sys/types.h>
#include <stdio.h>

#ifdef STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#elif HAVE_STRINGS_H
#include <strings.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif
#include <sys/time.h>

static void jabby_on_packet(void *udata, ikspak *pak);
static void jabby_on_log(void *udata, char *xml, int dir);

void jabby_register_to(jabby_session *ses, gchar *to, gchar *username,
                       gchar *password)
{
  iks *x, *y;

  x = iks_new("iq");
  iks_insert_attrib(x,"id","register_1");
  iks_insert_attrib(x,"type","set");
  iks_insert_attrib(x,"to", to);
  y = iks_insert(x, "query");
  iks_insert_attrib(y, "xmlns", IKS_NS_REGISTER);
  iks_insert_cdata(iks_insert(y, "username"), username, -1);
  iks_insert_cdata(iks_insert(y, "password"), password, -1);
//  iks_insert_cdata(iks_insert(y, "first"), "", -1);
//  iks_insert_cdata(iks_insert(y, "last"), "", -1);
//  iks_insert_cdata(iks_insert(y, "nick"), "", -1);
//  iks_insert_cdata(iks_insert(y, "city"), "", -1);
  iks_send(ses->parser, x);
  iks_delete(x);
}

void jabby_get_agents(jabby_session *ses)
{
  iks *x, *y;

  x = iks_new("iq");
  iks_insert_attrib(x,"id","agents");
  iks_insert_attrib(x,"type","get");
  iks_insert_attrib(x,"to",ses->id->server);
  y = iks_insert(x, "query");
  iks_insert_attrib(y, "xmlns", IKS_NS_AGENTS);
  iks_send(ses->parser, x);
  iks_delete(x);
}

jabby_session *jabby_session_new(gchar *id, gchar *pass)
{
  jabby_session *jabby_session;

  jabby_session = g_malloc(sizeof(*jabby_session));
  jabby_session->debug = TRUE;
  jabby_session->state = JABBY_NET_OFF;
  jabby_session->parser = iks_jabber_new(jabby_session, jabby_on_packet);
  if(!jabby_session->parser)
  {
    g_free(jabby_session);
    return NULL;
  }
  iks_set_logging(jabby_session->parser, jabby_on_log);
  jabby_session->id = iks_id_new(NULL, id);
  jabby_session->pass = g_strdup(pass);
  return jabby_session;
}

gint jabby_connect(jabby_session *ses)
{
  gint res;
  jabby_event *event;

  if (!ses)
  {
    g_print ("ses == NULL\n");
    return 0;
  }
  res = iks_connect_tcp(ses->parser, ses->id->server, 0);
  if(!res)
  {
    event = jabby_event_new(JABBY_EVENT_CONNECTION_FAILED);
    jabby_event_add(event);
    return 1;
  }
  ses->state = JABBY_NET_CONNECT;
  return 0;
}

jabby_event *jabby_event_next(jabby_session *ses)
{
  fd_set fds, fds2;
  int sock;
  struct timeval timeout;
  jabby_event *event;

  if(!ses)
    return NULL;
  event = jabby_event_get();
  if(event)
  {
    g_print("events in queue\n");
    return event;
  }
  sock = iks_fd(ses->parser);
  FD_ZERO(&fds);
  FD_ZERO(&fds2);
  FD_SET(0, &fds);
  if(sock != -1)
  {
    FD_SET(sock, &fds);
    if(ses->state == JABBY_NET_CONNECT) FD_SET(sock, &fds2);
  }

  timeout.tv_sec = 0;
  timeout.tv_usec = 0;
  if(select(sock + 2, &fds, &fds2, NULL, &timeout) > 0)
  {
    if(sock != -1 && (FD_ISSET(sock, &fds) || FD_ISSET(sock, &fds2)))
    {
      if (!iks_recv(ses->parser, 0))
      {
        g_print("JABBY_EVENT_DISCONNECT after iks_recv()\n");
        jabby_disconnect(ses);
        event = jabby_event_new(JABBY_EVENT_DISCONNECT);
        jabby_event_add(event);
      }
    }
  }
  return jabby_event_get();
}

void jabby_disconnect(jabby_session *ses)
{
  iks_disconnect(ses->parser);
  ses->state = JABBY_NET_OFF;
}

void jabby_get_roster(jabby_session *ses)
{
  iks *x, *y;

//  x = iks_make_iq (IKS_TYPE_GET, IKS_NS_ROSTER);
  x = iks_new("iq");
  iks_insert_attrib(x,"id","roster_1");
  iks_insert_attrib(x,"type","get");
  y = iks_insert(x, "query");
  iks_insert_attrib(y, "xmlns", IKS_NS_ROSTER);
  iks_send(ses->parser, x);
  iks_delete(x);
}

gint jabby_presence(jabby_session *ses, enum jabby_presence_type type,
                    gchar *message, gchar *to)
{
  iks *x;

  if (type == JABBY_PRESENCE_UNAVAILABLE)
    x = iks_make_pres (IKS_TYPE_UNAVAILABLE, 0, to, message);
  else
  {
    if(type == JABBY_PRESENCE_INVISIBLE)
      x = iks_make_pres (IKS_TYPE_INVISIBLE, type, to, message);
    else
      x = iks_make_pres (IKS_TYPE_AVAILABLE, type, to, message);
    if(ses->state == JABBY_NET_AUTH)
      ses->state = JABBY_NET_ON;
  }
  iks_send(ses->parser, x);
  iks_delete(x);
  return 0;
}

gint jabby_contact_add(jabby_session *ses, gchar *name, gchar *jid,
                       gchar *group)
{
  iks *x, *y, *z, *g;

  x = iks_new("iq");
  iks_insert_attrib(x,"type","set");
  y = iks_insert(x,"query");
  iks_insert_attrib(y,"xmlns",IKS_NS_ROSTER);
  z = iks_insert(y, "item");
  iks_insert_attrib(z,"jid",jid);
  iks_insert_attrib(z,"name",name);
  g = iks_insert(z, "group");
  iks_insert_cdata(g, group, -1);
  iks_send(ses->parser, x);
  iks_delete(x);
  return 0;
}

gint jabby_contact_remove(jabby_session *ses, gchar *jid)
{
  iks *x, *y, *z;

  x = iks_new("iq");
  iks_insert_attrib(x,"type","set");
  y = iks_insert(x,"query");
  iks_insert_attrib(y,"xmlns",IKS_NS_ROSTER);
  z = iks_insert(y,"item");
  iks_insert_attrib(z,"jid",jid);
  iks_insert_attrib(z,"subscription","remove");
  iks_send(ses->parser, x);
  iks_delete(x);
  return 0;
}

gint jabby_subscribe_ask(jabby_session *ses, gchar *jid, gchar *status)
{
  iks *x, *y;

  x = iks_new("presence");
  iks_insert_attrib(x,"to",jid);
  iks_insert_attrib(x,"type","subscribe");
  y = iks_insert(x,"status");
  if(status)
    iks_insert_cdata(y,status,-1);
  else
    iks_insert_cdata(y,"I would like to add you to my roster.",-1);
  iks_send(ses->parser,x);
  iks_delete(x);
  return 0;
}

gint jabby_subscribe_accept(jabby_session *ses, gchar *jid)
{
  iks *x;

  x = iks_new("presence");
  iks_insert_attrib(x,"to",jid);
  iks_insert_attrib(x,"type","subscribed");
  iks_send(ses->parser,x);
  iks_delete(x);
  return 0;
}

gint jabby_subscribe_deny(jabby_session *ses, gchar *jid)
{
  iks *x;

  x = iks_new("presence");
  iks_insert_attrib(x,"to",jid);
  iks_insert_attrib(x,"type","unsubscribed");
  iks_send(ses->parser,x);
  iks_delete(x);
  return 0;
}

gint jabby_unsubscribe(jabby_session *ses, gchar *jid)
{
  iks *x;

  x = iks_new("presence");
  iks_insert_attrib(x,"to",jid);
  iks_insert_attrib(x,"type","unsubscribe");
  iks_send(ses->parser,x);
  iks_delete(x);
  return 0;
}

gint jabby_message(jabby_session *ses, gchar *to, gchar *id, gchar *message,
                   gchar *subject, enum jabby_message_type type)
{
  iks *x, *y;
  gchar *from;

  x = iks_new("message");
  iks_insert_attrib(x, "to", to);
  from = g_strdup_printf("%s@%s/%s",ses->id->user, ses->id->server,ses->id->resource);
  iks_insert_attrib(x, "from", from);
  g_free(from);
  if(id)
    iks_insert_attrib(x, "id", id);
  switch(type)
  {
    case JABBY_MESSAGE_CHAT:
      iks_insert_attrib(x, "type", "chat");
      break;
    case JABBY_MESSAGE_ERROR:
      iks_insert_attrib(x, "type", "error");
      y = iks_insert(x, "error");
      iks_insert_cdata(y, message, -1);//FIXME
      break;
    default:
      break;
  }
  if(subject)
  {
    y = iks_insert(x, "subject");
    iks_insert_cdata(y, subject, -1);
  }
  y = iks_insert(x, "body");
  iks_insert_cdata(y, message, -1);
  iks_send(ses->parser, x);
  iks_delete(x);
  return 0;
}

void jabby_session_free(jabby_session *ses)
{
  if(!ses)
    return;
  iks_id_delete(ses->id);
  g_free(ses->pass);
  iks_parser_delete(ses->parser);
  g_free(ses);
}


/*
   *
   * functions not to use by the user
   *
*/

static
void jabby_on_packet(void *udata, ikspak *pak)
{
  jabby_session *ses;

  g_print("on packet\n");
  ses = (jabby_session*) udata;
  if(!pak)//FIXME
  {
    jabby_event *event;
    g_print("JABBY_EVENT_DISCONNECT after !pak\n");
    jabby_disconnect(ses);
    event = jabby_event_new(JABBY_EVENT_DISCONNECT);
    jabby_event_add(event);
    return;
  }

  switch(pak->type)
  {
    case IKS_PAK_STREAM:
      {
        iks *x;
        int res;

        ses->state = JABBY_NET_STREAM;
        x = iks_make_auth(ses->id, ses->pass, iks_find_attrib(pak->x, "id"));
        iks_insert_attrib(x, "id", "auth");
        res = iks_send(ses->parser, x);
        iks_delete(x);
        if(!res)
        {
          jabber_reconnect(ses); // is it OK?
          break;
        }
        if(ses->debug)
          g_print("Connected, logging in...\n");
        break;
      }

    case IKS_PAK_MESSAGE:
      {
        g_print("packet: message\n");
        jabby_parse_message(ses, pak);
      }
      break;

    case IKS_PAK_PRESENCE:
      {
        g_print("packet: presence\n");
        jabby_parse_presence(ses, pak);
      }
      break;

    case IKS_PAK_S10N:
      {

        g_print("packet: s10n\n");
        jabby_parse_presence(ses, pak);
      }
      break;

    case IKS_PAK_IQ:
      {
        g_print("packet: iq\n");
        jabby_parse_iq(ses, pak);
      }
      break;

    case IKS_PAK_ERROR:
      if( iks_has_children(pak->x) && iks_cdata(iks_child(pak->x)) )
        g_print("Stream:error: %s\n",iks_cdata(iks_child(pak->x)));
      else
        g_print("Stream:error without error-text.\n");
      break;

    case IKS_PAK_UNKNOWN:
      g_print("Stream:Got Unknown PAK\n");
      break;

  }
}

static
void jabby_on_log(void *udata, char *xml, int dir)
{
  jabby_session *ses;

  g_print("on log\n");

  ses = (jabby_session*) udata;
  if(!(ses->debug))
    return;
  if(dir == IKS_DIR_IN)
    g_print("data received:\n");
  else
    g_print("data send:\n");
  g_print("%s\n\n", xml);
}

